/*
 * =============================================================================
 * 
 *   Copyright (c) 2011-2013, The THYMELEAF team (http://www.thymeleaf.org)
 * 
 *   Licensed under the Apache License, Version 2.0 (the "License");
 *   you may not use this file except in compliance with the License.
 *   You may obtain a copy of the License at
 * 
 *       http://www.apache.org/licenses/LICENSE-2.0
 * 
 *   Unless required by applicable law or agreed to in writing, software
 *   distributed under the License is distributed on an "AS IS" BASIS,
 *   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *   See the License for the specific language governing permissions and
 *   limitations under the License.
 * 
 * =============================================================================
 */
package org.thymeleaf.spring4.processor.attr;

import java.util.Arrays;

import org.thymeleaf.Arguments;
import org.thymeleaf.dom.Attribute;
import org.thymeleaf.dom.Element;
import org.thymeleaf.exceptions.TemplateProcessingException;
import org.thymeleaf.spring4.util.FieldUtils;
import org.thymeleaf.standard.processor.attr.AbstractStandardSingleAttributeModifierAttrProcessor;
import org.thymeleaf.util.StringUtils;

/**
 * Adds the given class to the field on which this attribute is applied, if that
 * field contains errors.  It's similar to a combination of <tt>th:classappend</tt>
 * with a <tt>${#fields.hasErrors()}</tt> expression.
 * 
 * @author Daniel Fern&aacute;ndez
 * @since 2.1.0
 */
public final class SpringErrorClassAttrProcessor
        extends AbstractStandardSingleAttributeModifierAttrProcessor {

    public static final int ATTR_PRECEDENCE = 1500;
    public static final String ATTR_NAME = "errorclass";
    public static final String TARGET_ATTR_NAME = "class";



    public SpringErrorClassAttrProcessor() {
        super(ATTR_NAME);
    }



    @Override
    public int getPrecedence() {
        return ATTR_PRECEDENCE;
    }





    @Override
    protected String getTargetAttributeName(
            final Arguments arguments, final Element element, final String attributeName) {
        return TARGET_ATTR_NAME;
    }


    @Override
    protected String getTargetAttributeValue(
            final Arguments arguments, final Element element, final String attributeName) {

        final String fieldName = element.getAttributeValueFromNormalizedName("name");
        if (StringUtils.isEmptyOrWhitespace(fieldName)) {
            final String[] fieldProcessorNames =
                    Attribute.applyPrefixToAttributeName(
                            AbstractSpringFieldAttrProcessor.ATTR_NAME, Attribute.getPrefixFromAttributeName(attributeName));
            throw new TemplateProcessingException(
                    "Cannot apply \"" + attributeName + "\": this attribute requires the existence of " +
                    "a \"name\" (or " + Arrays.asList(fieldProcessorNames) + ") attribute with non-empty " +
                    "value in the same host tag.");
        }

        // If the selected field has errors, append the selected class
        if (FieldUtils.hasErrors(arguments.getConfiguration(), arguments, fieldName)) {

            // Compute the CSS class to be applied exactly as it would in a normal th:classappend processor
            return super.getTargetAttributeValue(arguments, element, attributeName);

        }

        return "";

    }





    @Override
    protected ModificationType getModificationType(
            final Arguments arguments, final Element element, final String attributeName, final String newAttributeName) {
        return ModificationType.APPEND_WITH_SPACE;
    }





    @Override
    protected boolean removeAttributeIfEmpty(
            final Arguments arguments, final Element element, final String attributeName, final String newAttributeName) {
        return true;
    }



}
