/*
 * Copyright 2011 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.springframework.social.oauth1;

/**
 * A OAuth 1.0 request token that has been authorized by the user.
 * Constructed after the user grants the consumer application access to their data hosted at the service provider.
 * This is typically achieved by the user clicking "Allow" over at the provider's site.
 * The service provider returns a Verifier string in the authorization callback that must also be submitted in the access token request.
 * @author Keith Donald
 */
public class AuthorizedRequestToken {
	
	private final OAuthToken requestToken;
	
	private final String verifier;

	/**
	 * Creates an authorized request token.
	 * @param requestToken the request token object
	 */
	public AuthorizedRequestToken(OAuthToken requestToken, String verifier) {
		this.requestToken = requestToken;
		this.verifier = verifier;
	}
	
	/**
	 * The request token value.
	 */
	public String getValue() {
		return requestToken.getValue();
	}

	/**
	 * The request token secret.
	 */
	public String getSecret() {
		return requestToken.getSecret();
	}
	
	/**
	 * The verifier string generated by the provider.
	 */
	public String getVerifier() {
		return verifier;
	}
	
}