
/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.server.schema.bootstrap;


import java.util.ArrayList;
import javax.naming.NamingException;

import org.apache.directory.server.schema.registries.Registries;
import org.apache.directory.shared.ldap.schema.UsageEnum;


/**
 * A producer of schema attributeType definations for the apache schema.  This
 * code has been automatically generated using schema files in the OpenLDAP
 * format along with the directory plugin for maven.  This has been done
 * to facilitate Eve<->OpenLDAP schema interoperability.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class ApacheAttributeTypeProducer extends AbstractBootstrapProducer
{

    public ApacheAttributeTypeProducer()
    {
        super( ProducerTypeEnum.ATTRIBUTE_TYPE_PRODUCER );
    }


    // ------------------------------------------------------------------------
    // BootstrapProducer Methods
    // ------------------------------------------------------------------------


    /**
     * @see BootstrapProducer#produce( Registries, ProducerCallback )
     */
    public void produce( Registries registries, ProducerCallback cb )
        throws NamingException
    {
        ArrayList<String> names = new ArrayList<String>();
        BootstrapAttributeType attributeType;


        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.1 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.1", registries );
        attributeType.setDescription( "Index attribute DN whose values are normalized based on schema" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "exactDnAsStringMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "apacheNdn" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.1", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.2 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.2", registries );
        attributeType.setDescription( "Index attribute for DN whose values are NOT normalized in any way" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "exactDnAsStringMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "apacheUpdn" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.2", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.3 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.3", registries );
        attributeType.setDescription( "Index attribute used to track the existence of attributes" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setSuperiorId( "name" );
        attributeType.setEqualityId( "caseIgnoreMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.15" );

        names.clear();
        names.add( "apacheExistence" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.3", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.4 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.4", registries );
        attributeType.setDescription( "Index attribute used to track one level searches" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "integerMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.27" );

        names.clear();
        names.add( "apacheOneLevel" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.4", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.5 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.5", registries );
        attributeType.setDescription( "Index attribute used to track single level aliases" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "integerMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.27" );

        names.clear();
        names.add( "apacheOneAlias" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.5", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.6 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.6", registries );
        attributeType.setDescription( "Index attribute used to track sub level aliases" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "integerMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.27" );

        names.clear();
        names.add( "apacheSubAlias" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.6", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.43 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.43", registries );
        attributeType.setDescription( "Index attribute used to track sub level searches" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "integerMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.27" );

        names.clear();
        names.add( "apacheSubLevel" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.43", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.7 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.7", registries );
        attributeType.setDescription( "asdf" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setSuperiorId( "distinguishedName" );
        attributeType.setEqualityId( "distinguishedNameMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "apacheAlias" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.7", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.9 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.9", registries );
        attributeType.setDescription( "Single-use Authentication Mechanism type/vendor code" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "integerMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.27" );

        names.clear();
        names.add( "apacheSamType" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.9", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.8 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.8", registries );
        attributeType.setDescription( "Attribute to describe the name of a Java Preferences API node" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.15" );

        names.clear();
        names.add( "prefNodeName" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.8", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.10 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.10", registries );
        attributeType.setDescription( "Used to track a subentry associated with an autonomousArea" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setSuperiorId( "distinguishedName" );
        attributeType.setEqualityId( "distinguishedNameMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "autonomousAreaSubentry" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.10", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.11 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.11", registries );
        attributeType.setDescription( "Used to track a subentry associated with access control areas" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setSuperiorId( "distinguishedName" );
        attributeType.setEqualityId( "distinguishedNameMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "accessControlSubentries" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.11", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 2.5.24.4 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "2.5.24.4", registries );
        attributeType.setDescription( "Access control information that applies to a set of entries" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "directoryStringFirstComponentMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.1" );

        names.clear();
        names.add( "prescriptiveACI" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "2.5.24.4", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 2.5.24.5 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "2.5.24.5", registries );
        attributeType.setDescription( "Access control information that applies to a single entry" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "directoryStringFirstComponentMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.1" );

        names.clear();
        names.add( "entryACI" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "2.5.24.5", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 2.5.24.6 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "2.5.24.6", registries );
        attributeType.setDescription( "Access control information that applies to a single subentry" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "directoryStringFirstComponentMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.1" );

        names.clear();
        names.add( "subentryACI" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "2.5.24.6", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.15 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.15", registries );
        attributeType.setDescription( "A string up to 256 characters in length" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 256 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseIgnoreIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "apacheServicePid" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.15", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.16 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.16", registries );
        attributeType.setDescription( "A string up to 256 characters in length" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 256 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseIgnoreIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "apacheServiceFactoryPid" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.16", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.17 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.17", registries );
        attributeType.setDescription( "A string up to 256 characters in length" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 256 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseIgnoreIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "apacheCatalogEntryName" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.17", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.18 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.18", registries );
        attributeType.setDescription( "A string up to 256 characters in length" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 256 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseIgnoreIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "apacheCatalogEntryBaseDn" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.18", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.19 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.19", registries );
        attributeType.setDescription( "A windows file path where case does not make a difference" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseIgnoreIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "windowsFilePath" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.19", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.20 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.20", registries );
        attributeType.setDescription( "A UNIX file path where case does make a difference" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "unixFilePath" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.20", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.21 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.21", registries );
        attributeType.setDescription( "The fully qualified name for a (Java) class" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "fullyQualifiedJavaClassName" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.21", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.22 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.22", registries );
        attributeType.setDescription( "The actual byte code for a (Java) class" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.5" );

        names.clear();
        names.add( "javaClassByteCode" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.22", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.23 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.23", registries );
        attributeType.setDescription( "The default search context for the Ldap Class Loader" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setSuperiorId( "distinguishedName" );
        attributeType.setEqualityId( "distinguishedNameMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "classLoaderDefaultSearchContext" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.23", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.5.1 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.5.1", registries );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "storedProcLangId" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.5.1", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.5.2 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.5.2", registries );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "storedProcUnitName" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.5.2", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.5.4 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.5.4", registries );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.5" );

        names.clear();
        names.add( "javaByteCode" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.5.4", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.5.6 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.5.6", registries );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "javaxScriptLangId" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.5.6", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.5.7 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.5.7", registries );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "javaxScriptCode" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.5.7", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.25 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.25", registries );
        attributeType.setDescription( "Trigger specification that applies to a set of entries" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "directoryStringFirstComponentMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.59" );

        names.clear();
        names.add( "prescriptiveTriggerSpecification" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.25", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.26 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.26", registries );
        attributeType.setDescription( "Trigger specification that applies to a single entry" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "directoryStringFirstComponentMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.59" );

        names.clear();
        names.add( "entryTriggerSpecification" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.26", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.27 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.27", registries );
        attributeType.setDescription( "Used to track subentries associated with a trigger area which an entry falls under" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setSuperiorId( "distinguishedName" );
        attributeType.setEqualityId( "distinguishedNameMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "triggerExecutionSubentries" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.27", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.1.16.4 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.1.16.4", registries );
        attributeType.setDescription( "UUID of the entry" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "uuidMatch" );
        attributeType.setOrderingId( "uuidOrderingMatch" );
        attributeType.setSyntaxId( "1.3.6.1.1.16.1" );

        names.clear();
        names.add( "entryUUID" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.1.16.4", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.4203.666.1.7 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.4203.666.1.7", registries );
        attributeType.setDescription( "change sequence number of the entry" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 64 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "CSNMatch" );
        attributeType.setOrderingId( "CSNOrderingMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.4203.666.11.2.1" );

        names.clear();
        names.add( "entryCSN" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.4203.666.1.7", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.4203.666.1.25 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.4203.666.1.25", registries );
        attributeType.setDescription( "the largest committed CSN of a context" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 64 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "CSNMatch" );
        attributeType.setOrderingId( "CSNOrderingMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.4203.666.11.2.1" );

        names.clear();
        names.add( "contextCSN" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.4203.666.1.25", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.31 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.31", registries );
        attributeType.setDescription( "Whether or not an entry has been deleted." );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "booleanMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.7" );

        names.clear();
        names.add( "entryDeleted" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.31", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.32 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.32", registries );
        attributeType.setDescription( "A multivalued comparator description attribute" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "caseIgnoreMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.15" );

        names.clear();
        names.add( "comparators" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.32", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.33 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.33", registries );
        attributeType.setDescription( "A multivalued normalizer description attribute" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "caseIgnoreMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.15" );

        names.clear();
        names.add( "normalizers" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.33", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.34 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.34", registries );
        attributeType.setDescription( "A multivalued syntaxCheckers description attribute" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DIRECTORY_OPERATION" ) );
        attributeType.setEqualityId( "caseIgnoreMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.15" );

        names.clear();
        names.add( "syntaxCheckers" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.34", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.35 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.35", registries );
        attributeType.setDescription( "time which schema was modified" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setSuperiorId( "modifyTimestamp" );

        names.clear();
        names.add( "schemaModifyTimestamp" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.35", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.36 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.36", registries );
        attributeType.setDescription( "the DN of the modifier of the schema" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setSuperiorId( "modifiersName" );

        names.clear();
        names.add( "schemaModifiersName" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.36", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.37 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.37", registries );
        attributeType.setDescription( "the DN of the schema subentry the modification info corresponds to" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 32768 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "distinguishedNameMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.15" );

        names.clear();
        names.add( "subschemaSubentryName" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.37", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.38 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.38", registries );
        attributeType.setDescription( "The format of the private key used for TLS" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "privateKeyFormat" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.38", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.41 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.41", registries );
        attributeType.setDescription( "The format of the public key used for TLS" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "publicKeyFormat" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.41", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.39 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.39", registries );
        attributeType.setDescription( "The algorithm used for the key/pair used by the server for TLS" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setEqualityId( "caseExactIA5Match" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.26" );

        names.clear();
        names.add( "keyAlgorithm" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.39", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.40 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.40", registries );
        attributeType.setDescription( "The private key material used for TLS" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.5" );

        names.clear();
        names.add( "privateKey" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.40", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.42 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.42", registries );
        attributeType.setDescription( "The public key material used for TLS" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "USER_APPLICATIONS" ) );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.5" );

        names.clear();
        names.add( "publicKey" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.42", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.44 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.44", registries );
        attributeType.setDescription( "Revision numbers used in change log" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( false );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "integerMatch" );
        attributeType.setOrderingId( "integerOrderingMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.27" );

        names.clear();
        names.add( "revisions" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.44", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.45 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.45", registries );
        attributeType.setDescription( "represents the time when the change event occurred" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "generalizedTimeMatch" );
        attributeType.setOrderingId( "generalizedTimeOrderingMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.24" );

        names.clear();
        names.add( "changeTime" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.45", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.46 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.46", registries );
        attributeType.setDescription( "type of change operation" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "caseIgnoreMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.44" );

        names.clear();
        names.add( "changeType" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.46", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.47 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.47", registries );
        attributeType.setDescription( "the unique sequential id for the event (a.k.a revision number)" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "integerMatch" );
        attributeType.setOrderingId( "integerOrderingMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.27" );

        names.clear();
        names.add( "rev" );
        names.add( "eventId" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.47", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.48 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.48", registries );
        attributeType.setDescription( "the principal committing the change" );
        attributeType.setCanUserModify( ! true );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setEqualityId( "exactDnAsStringMatch" );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "committer" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.48", attributeType );

        // --------------------------------------------------------------------
        // AttributeType 1.3.6.1.4.1.18060.0.4.1.2.49 
        // --------------------------------------------------------------------

        attributeType = newAttributeType( "1.3.6.1.4.1.18060.0.4.1.2.49", registries );
        attributeType.setDescription( "tells about the changelog context suffix" );
        attributeType.setCanUserModify( ! false );
        attributeType.setSingleValue( true );
        attributeType.setCollective( false );
        attributeType.setObsolete( false );
        attributeType.setLength( 0 );
        attributeType.setUsage( UsageEnum.getUsage( "DSA_OPERATION" ) );
        attributeType.setSyntaxId( "1.3.6.1.4.1.1466.115.121.1.12" );

        names.clear();
        names.add( "changeLogContext" );
        attributeType.setNames( names.toArray( EMPTY ) );
        cb.schemaObjectProduced( this, "1.3.6.1.4.1.18060.0.4.1.2.49", attributeType );
    }
}
