/*
 * Decompiled with CFR 0.152.
 */
package org.apache.crunch.io.text.csv;

import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import javax.annotation.ParametersAreNonnullByDefault;
import org.apache.hadoop.io.Text;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ParametersAreNonnullByDefault
public class CSVLineReader {
    private static final Logger LOGGER = LoggerFactory.getLogger(CSVLineReader.class);
    public static final int DEFAULT_BUFFER_SIZE = 65536;
    private final InputStreamReader inputStreamReader;
    private final String inputFileEncoding;
    private final CharsetEncoder charsetEncoder;
    private char[] buffer;
    private final int bufferSize;
    private int bufferLength = 0;
    private int bufferPosition = 0;
    private boolean bufferIsPadded = false;
    private static char CR = (char)13;
    private static char LF = (char)10;
    private boolean endOfFile = false;
    public static final char DEFAULT_QUOTE_CHARACTER = '\"';
    public static final char DEFAULT_ESCAPE_CHARACTER = '\\';
    public static final char NULL_CHARACTER = '\u0000';
    public static final String DEFAULT_INPUT_FILE_ENCODING = "UTF-8";
    public static final int DEFAULT_MAXIMUM_RECORD_SIZE = 0x4000000;
    private final int maximumRecordSize;
    private final char openQuoteChar;
    private final char closeQuoteChar;
    private final char escape;
    private boolean inMultiLine = false;
    private boolean currentlyInQuotes = false;
    private boolean endOfLineReached = false;
    private Text inputText = new Text();

    public CSVLineReader(InputStream inputStream) throws UnsupportedEncodingException {
        this(inputStream, 65536, DEFAULT_INPUT_FILE_ENCODING, '\"', '\"', '\\', 0x4000000);
    }

    public CSVLineReader(InputStream inputStream, int bufferSize, String inputFileEncoding, char openQuoteChar, char closeQuoteChar, char escapeChar, int maximumRecordSize) {
        Preconditions.checkNotNull((Object)inputStream, (Object)"inputStream may not be null");
        Preconditions.checkNotNull((Object)inputFileEncoding, (Object)"inputFileEncoding may not be null");
        if (bufferSize <= 0) {
            throw new IllegalArgumentException("The buffer (" + bufferSize + ")cannot be <= 0");
        }
        try {
            this.inputStreamReader = new InputStreamReader(inputStream, inputFileEncoding);
        }
        catch (UnsupportedEncodingException uee) {
            throw new RuntimeException(inputFileEncoding + " is not a supported encoding.", uee);
        }
        this.bufferSize = bufferSize;
        this.buffer = new char[this.bufferSize];
        if (this.isSameCharacter(openQuoteChar, escapeChar)) {
            throw new IllegalArgumentException("The open quote (" + openQuoteChar + ") and escape (" + escapeChar + ") characters must be different!");
        }
        if (this.isSameCharacter(closeQuoteChar, escapeChar)) {
            throw new IllegalArgumentException("The close quote (" + closeQuoteChar + ") and escape (" + escapeChar + ") characters must be different!");
        }
        this.openQuoteChar = openQuoteChar;
        this.closeQuoteChar = closeQuoteChar;
        this.escape = escapeChar;
        this.inputFileEncoding = inputFileEncoding;
        this.charsetEncoder = Charset.forName(inputFileEncoding).newEncoder();
        this.maximumRecordSize = maximumRecordSize;
    }

    public int readCSVLine(Text input) throws IOException {
        Preconditions.checkNotNull((Object)input, (Object)"inputText may not be null");
        this.inputText = new Text(input);
        long totalBytesConsumed = 0L;
        if (this.endOfFile) {
            return 0;
        }
        if (this.inMultiLine) {
            throw new RuntimeException("Cannot begin reading a CSV record while inside of a multi-line CSV record.");
        }
        StringBuilder stringBuilder = new StringBuilder();
        do {
            this.inputText.clear();
            totalBytesConsumed += (long)this.readFileLine(this.inputText);
            stringBuilder.append(this.inputText.toString());
            if (!this.currentlyInQuotes || this.endOfFile) continue;
            stringBuilder.append('\n');
            if (totalBytesConsumed <= (long)this.maximumRecordSize && totalBytesConsumed <= Integer.MAX_VALUE) continue;
            String record = stringBuilder.toString();
            LOGGER.error("Possibly malformed file encountered. First line of record: " + record.substring(0, record.indexOf(10)));
            throw new IOException("Possibly malformed file encountered. Check log statements for more information");
        } while (this.currentlyInQuotes && !this.endOfFile);
        input.set(stringBuilder.toString());
        return (int)totalBytesConsumed;
    }

    public int readFileLine(Text input) throws IOException {
        Preconditions.checkNotNull((Object)input, (Object)"inputText may not be null");
        if (this.endOfFile) {
            return 0;
        }
        int newlineLength = 0;
        int inputTextLength = 0;
        long bytesConsumed = 0L;
        int readTextLength = 0;
        int startPosition = this.bufferPosition;
        this.endOfLineReached = false;
        this.inputText = new Text(input);
        do {
            boolean checkForLF = false;
            if (this.bufferPosition >= this.bufferLength) {
                this.refillBuffer();
                startPosition = this.bufferPosition;
                if (this.endOfFile) break;
            }
            newlineLength = 0;
            while (this.bufferPosition < this.bufferLength) {
                bytesConsumed += (long)this.calculateCharacterByteLength(this.buffer[this.bufferPosition]);
                if (this.buffer[this.bufferPosition] == this.escape) {
                    if (this.isNextCharacterEscapable(this.currentlyInQuotes, this.bufferPosition)) {
                        ++this.bufferPosition;
                        bytesConsumed += (long)this.calculateCharacterByteLength(this.buffer[this.bufferPosition]);
                    }
                } else if (this.buffer[this.bufferPosition] == this.openQuoteChar || this.buffer[this.bufferPosition] == this.closeQuoteChar) {
                    this.currentlyInQuotes = !this.currentlyInQuotes;
                } else if (this.buffer[this.bufferPosition] == LF || this.buffer[this.bufferPosition] == CR) {
                    boolean lastCharWasCR = this.buffer[this.bufferPosition] == CR;
                    this.endOfLineReached = true;
                    ++newlineLength;
                    ++this.bufferPosition;
                    if (lastCharWasCR && this.buffer[this.bufferPosition] == LF) {
                        lastCharWasCR = false;
                        ++newlineLength;
                        ++this.bufferPosition;
                        bytesConsumed += (long)this.calculateCharacterByteLength(this.buffer[this.bufferPosition]);
                        break;
                    }
                    if (!lastCharWasCR || this.bufferPosition < this.bufferLength) break;
                    checkForLF = true;
                    break;
                }
                ++this.bufferPosition;
            }
            if ((readTextLength = this.bufferPosition - startPosition - newlineLength) > Integer.MAX_VALUE - inputTextLength) {
                readTextLength = Integer.MAX_VALUE - inputTextLength;
            }
            if (readTextLength > 0) {
                this.inputText.set(this.inputText.toString() + new String(this.buffer, startPosition, readTextLength));
                inputTextLength += readTextLength;
            }
            if (!checkForLF) continue;
            this.refillBuffer();
            if (this.endOfFile) break;
            if (this.buffer[this.bufferPosition] != LF) continue;
            bytesConsumed += (long)this.calculateCharacterByteLength(this.buffer[this.bufferPosition]);
            ++this.bufferPosition;
            ++newlineLength;
        } while (newlineLength == 0 && bytesConsumed < Integer.MAX_VALUE);
        if (this.endOfLineReached) {
            this.inMultiLine = this.currentlyInQuotes;
        }
        if (bytesConsumed > Integer.MAX_VALUE) {
            throw new IOException("Too many bytes consumed before newline: 2147483647");
        }
        input.set(this.inputText);
        return (int)bytesConsumed;
    }

    public boolean isInMultiLine() {
        return this.inMultiLine;
    }

    public void resetMultiLine() {
        this.inMultiLine = false;
        this.currentlyInQuotes = false;
    }

    private boolean isSameCharacter(char c1, char c2) {
        return c1 != '\u0000' && c1 == c2;
    }

    private boolean isNextCharacterEscapable(boolean inQuotes, int i) {
        return inQuotes && this.buffer.length > i + 1 && (this.buffer[i + 1] == this.closeQuoteChar || this.buffer[i + 1] == this.openQuoteChar || this.buffer[i + 1] == this.escape);
    }

    private void refillBuffer() throws IOException {
        this.bufferPosition = 0;
        if (this.bufferIsPadded) {
            this.buffer = new char[this.bufferLength];
            this.bufferIsPadded = false;
        }
        this.bufferLength = this.inputStreamReader.read(this.buffer, 0, this.buffer.length);
        if (this.buffer.length == this.bufferLength) {
            char[] biggerBuffer = new char[this.bufferLength + 3];
            for (int i = 0; i < this.bufferLength; ++i) {
                biggerBuffer[i] = this.buffer[i];
            }
            this.buffer = biggerBuffer;
            this.bufferIsPadded = true;
        }
        if (this.bufferLength <= 0) {
            this.endOfFile = true;
        }
    }

    private int calculateCharacterByteLength(char character) {
        try {
            return this.charsetEncoder.encode(CharBuffer.wrap(new char[]{character})).limit();
        }
        catch (CharacterCodingException e) {
            throw new RuntimeException("The character attempting to be read (" + character + ") could not be encoded with " + this.inputFileEncoding);
        }
    }
}

