// Copyright 2008 Google Inc. All Rights Reserved.

package com.google.monitoring.runtime.instrumentation;

import org.objectweb.asm.ClassAdapter;
import org.objectweb.asm.ClassWriter;
import org.objectweb.asm.MethodVisitor;
import org.objectweb.asm.commons.CodeSizeEvaluator;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This is a class writer that gets used in place of the existing 
 * {@link ClassWriter}, and verifies properties of the class getting written.
 * 
 * Currently, it only checks to see if the methods are of the correct length
 * for Java methods (<64K).
 * 
 * @author jeremymanson@google.com (Jeremy Manson)
 */
public class VerifyingClassAdapter extends ClassAdapter {
  private static final Logger logger =
    Logger.getLogger(VerifyingClassAdapter.class.getName());

  /**
   * An enum which indicates whether the class in question is verified.
   */
  public enum State {
    PASS, UNKNOWN, FAIL_TOO_LONG;
  }
  
  final ClassWriter cw;
  final byte [] original;
  final String className;
  String message;
  State state;
  
  /**
   * @param cw A class writer that is wrapped by this class adapter
   * @param original the original bytecode
   * @param className the name of the class being examined.
   */
  public VerifyingClassAdapter(ClassWriter cw, byte [] original,
      String className) {
    super(cw);
    state = State.UNKNOWN;
    message = "The class has not finished being examined";
    this.cw = cw;
    this.original = original;
    this.className = className.replace('/', '.');
  }

  /**
   * {@inheritDoc}
   * 
   * In addition, the returned {@link MethodVisitor} will throw an exception
   * if the method is greater than 64K in length.
   */
  @Override
  public MethodVisitor visitMethod(
      final int access,
      final String name,
      final String desc,
      final String signature,
      final String[] exceptions) {
    MethodVisitor mv =
      super.visitMethod(access, name, desc, signature, exceptions);
    return new CodeSizeEvaluator(mv) {
      @Override
      public void visitEnd() {
        super.visitEnd();
        if (getMaxSize() > 64 * 1024) {
          state = State.FAIL_TOO_LONG;
          message = "the method " + name + " was too long.";
        }
      }
    };
  }
  
  /**
   * {@inheritDoc}
   */
  @Override
  public void visitEnd() {
    super.visitEnd();
    if (state == State.UNKNOWN) {
      state = State.PASS;
    }
  }

  /**
   * Gets the verification state of this class.
   * 
   * @return true iff the class passed inspection.
   */
  public boolean isVerified() {
    return state == State.PASS;
  }
  
  /**
   * Returns the byte array that contains the byte code for this class.
   * 
   * @return a byte array.
   */
  public byte[] toByteArray() {
    if (state != State.PASS) {
      logger.log(Level.WARNING,
          "Failed to instrument class " + className + " because " + message);
      return original;
    }
    return cw.toByteArray();
  }
}
