/*
 * Copyright 2002-2019 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.alibaba.nacos.common.packagescan.util;

import com.alibaba.nacos.common.packagescan.resource.AntPathMatcher;
import com.alibaba.nacos.common.packagescan.resource.PathMatchingResourcePatternResolver;

import java.util.Comparator;
import java.util.Map;

/**
 * Copy from https://github.com/spring-projects/spring-framework.git, with less modifications
 * Strategy interface for {@code String}-based path matching.
 *
 * <p>Used by {@link PathMatchingResourcePatternResolver},
 *
 * <p>The default implementation is {@link AntPathMatcher}, supporting the
 * Ant-style pattern syntax.
 *
 * @author Juergen Hoeller
 * @see AntPathMatcher
 * @since 1.2
 */
public interface PathMatcher {

    /**
     * Does the given {@code path} represent a pattern that can be matched
     * by an implementation of this interface?
     *
     * <p>If the return value is {@code false}, then the {@link #match}
     * method does not have to be used because direct equality comparisons
     * on the static path Strings will lead to the same result.
     *
     * @param path the path to check
     * @return {@code true} if the given {@code path} represents a pattern
     */
    boolean isPattern(String path);

    /**
     * Match the given {@code path} against the given {@code pattern},
     * according to this PathMatcher's matching strategy.
     *
     * @param pattern the pattern to match against
     * @param path    the path to test
     * @return {@code true} if the supplied {@code path} matched,
     * {@code false} if it didn't
     */
    boolean match(String pattern, String path);

    /**
     * Match the given {@code path} against the corresponding part of the given
     * {@code pattern}, according to this PathMatcher's matching strategy.
     *
     * <p>Determines whether the pattern at least matches as far as the given base
     * path goes, assuming that a full path may then match as well.
     *
     * @param pattern the pattern to match against
     * @param path    the path to test
     * @return {@code true} if the supplied {@code path} matched,
     * {@code false} if it didn't
     */
    boolean matchStart(String pattern, String path);

    /**
     * Given a pattern and a full path, determine the pattern-mapped part.
     *
     * <p>This method is supposed to find out which part of the path is matched
     * dynamically through an actual pattern, that is, it strips off a statically
     * defined leading path from the given full path, returning only the actually
     * pattern-matched part of the path.
     *
     * <p>For example: For "myroot/*.html" as pattern and "myroot/myfile.html"
     * as full path, this method should return "myfile.html". The detailed
     * determination rules are specified to this PathMatcher's matching strategy.
     *
     * <p>A simple implementation may return the given full path as-is in case
     * of an actual pattern, and the empty String in case of the pattern not
     * containing any dynamic parts (i.e. the {@code pattern} parameter being
     * a static path that wouldn't qualify as an actual {@link #isPattern pattern}).
     * A sophisticated implementation will differentiate between the static parts
     * and the dynamic parts of the given path pattern.
     *
     * @param pattern the path pattern
     * @param path    the full path to introspect
     * @return the pattern-mapped part of the given {@code path}
     * (never {@code null})
     */
    String extractPathWithinPattern(String pattern, String path);

    /**
     * Given a pattern and a full path, extract the URI template variables. URI template
     * variables are expressed through curly brackets ('{' and '}').
     *
     * <p>For example: For pattern "/hotels/{hotel}" and path "/hotels/1", this method will
     * return a map containing "hotel" &rarr; "1".
     *
     * @param pattern the path pattern, possibly containing URI templates
     * @param path    the full path to extract template variables from
     * @return a map, containing variable names as keys; variables values as values
     */
    Map<String, String> extractUriTemplateVariables(String pattern, String path);

    /**
     * Given a full path, returns a {@link Comparator} suitable for sorting patterns
     * in order of explicitness for that path.
     *
     * <p>The full algorithm used depends on the underlying implementation,
     * but generally, the returned {@code Comparator} will
     * {@linkplain java.util.List#sort(Comparator) sort}
     * a list so that more specific patterns come before generic patterns.
     *
     * @param path the full path to use for comparison
     * @return a comparator capable of sorting patterns in order of explicitness
     */
    Comparator<String> getPatternComparator(String path);

    /**
     * Combines two patterns into a new pattern that is returned.
     *
     * <p>The full algorithm used for combining the two pattern depends on the underlying implementation.
     *
     * @param pattern1 the first pattern
     * @param pattern2 the second pattern
     * @return the combination of the two patterns
     * @throws IllegalArgumentException when the two patterns cannot be combined
     */
    String combine(String pattern1, String pattern2);

}
